/* eslint-disable prefer-destructuring */

const { spawn } = require('child_process');
const path = require('path');

const {
  app,
  BrowserWindow,
  dialog,
  // crashReporter,
  Menu,
  MenuItem,
  // protocol,
  shell,
  ipcMain
} = require('electron');
const Store = require('electron-store');
const WebSocket = require('ws');

const store = new Store();

app.allowRendererProcessReuse = true;

if (process.argv.length === 1) {
  spawn('VOX Universal.exe', [], {
    detached: true,
    stdio: 'ignore'
  }).unref();
  process.exit(0);

  return;
}

if (!app.requestSingleInstanceLock()) {
  app.quit();

  return;
}

process.env.ELECTRON_DISABLE_SECURITY_WARNINGS = true;

// crashReporter.start({
//   productName: 'ReVOX',
//   companyName: 'Coppertino',
//   submitURL:
//     'https://appmon.vox.rocks/api/4/minidump/?sentry_key=d4baff1f21114fc8a6d21ea29a90b19a',
//   uploadToServer: true,
// });

let webAddr = process.argv[2];
let wsAddr = webAddr;
let debug = false;
if (process.argv.length > 3) {
  wsAddr = process.argv[3];
  if (process.argv.length > 4) {
    debug = process.argv[4] === 'true';
  }
} else {
  const args = webAddr.split('|');
  webAddr = args[0];
  wsAddr = args[1];
  debug = args[2] === 'true';
}

const windows = {};

const defaultWindowOptions = {
  title: 'VOX Universal',
  titleBarStyle: 'hiddenInset',
  width: 1440,
  height: 800,
  minWidth: 1000,
  minHeight: 600,
  backgroundColor: '#292929',
  autoHideMenuBar: true,
  icon: path.join(__dirname, 'icon.png'),
  webPreferences: {
    nodeIntegration: false,
    contextIsolation: true, // protect against prototype pollution
    webSecurity: false,
    allowRunningInsecureContent: false,
    defaultEncoding: 'UTF-8',
    devTools: debug,
    backgroundThrottling: false,
    preload: path.join(app.getAppPath(), 'preload.js') // use a preload script
  }
};

const fileMenu = [
  process.argv.length === 1 ? { role: 'quit' } : { role: 'close' }
];
!debug && fileMenu.unshift({ role: 'reload' });
const mainMenu = Menu.buildFromTemplate([
  {
    role: 'appMenu'
  },
  {
    label: 'File',
    submenu: fileMenu
  },
  {
    role: 'editMenu'
  },
  {
    role: 'windowMenu'
  },
  {
    label: 'Help',
    submenu: [
      {
        label: 'Home page',
        click() {
          shell.openExternal('https://vox.rocks');
        }
      },
      {
        label: 'Forum',
        click() {
          shell.openExternal('https://vox.rocks/forum');
        }
      },
      {
        label: 'Support',
        click() {
          shell.openExternal('https://vox.rocks/forum/c/vox-tech-support');
        }
      }
    ]
  }
]);
debug && mainMenu.insert(3, new MenuItem({ role: 'viewMenu' }));
Menu.setApplicationMenu(mainMenu);

ipcMain.on('setStoreValue', (event, key, value) => {
  store.set(key, value);
});

ipcMain.on('getStoreValue', (event, key) => {
  event.returnValue = store.get(key);
});

ipcMain.on('clearStore', (event, key) => {
  store.clear();
});

ipcMain.on('appQuit', () => {
  app.quit();
});

ipcMain.on('getPath', (event, name) => {
  event.returnValue = app.getPath(name);
});

function createWindow(url, opts) {
  let w = new BrowserWindow({ ...defaultWindowOptions, ...opts });
  windows[w.id] = w;
  w.on('ready-to-show', () => w.show());

  ipcMain.handle('showMessageBox', async (event, props) => {
    return await dialog.showMessageBox(w, props);
  });

  ipcMain.on('showCurrentWindow', () => {
    w.show();
  });

  ipcMain.handle('openFileDialog', async (event, props) => {
    return await dialog.showOpenDialog(props);
  });

  w.loadURL(url);
  w.center();
  w.on('close', () => delete windows[w.id]);
  // eslint-disable-next-line no-return-assign
  w.on('closes', () => (w = null));
  w.webContents.on('preload-error', (evt, preloadPath, err) => {
    console.error(`preload file '${preloadPath}' load error: ${err}`);
  });
  w.webContents.session.setPermissionRequestHandler(
    (webContents, permission, done) => {
      const parsedUrl = new URL(webContents.getURL());
      const options = {
        title: 'Permission Request',
        message: `Allow '${parsedUrl.origin}' to access '${permission}'?`,
        buttons: ['OK', 'Cancel'],
        cancelId: 1
      };
      dialog
        .showMessageBox(w, options)
        .then(({ response }) => done(response === 0));
    }
  );

  return w;
}

app.on('ready', () => {
  if (debug) {
    try {
      const {
        default: installExtension,
        // REACT_DEVELOPER_TOOLS,
        MOBX_DEVTOOLS,
        REDUX_DEVTOOLS
        // eslint-disable-next-line global-require
      } = require('electron-devtools-installer');

      [/*REACT_DEVELOPER_TOOLS,*/ MOBX_DEVTOOLS, REDUX_DEVTOOLS].forEach(
        extensionRef => {
          installExtension(extensionRef)
            .then(name => console.log(`Added Extension:  ${name}`))
            .catch(err => console.log('An error occurred: ', err));
        }
      );
    } catch (e) {
      console.warn(e);
    }
  }
  const mainWindow = createWindow(`http://${webAddr}/?ws=${wsAddr}`, {});
  mainWindow.maximize();
  debug && mainWindow.webContents.openDevTools();
});

app.on('activate', () => {
  // On macOS it's common to re-create a window in the app when the
  // dock icon is clicked and there are no other windows open.
  if (BrowserWindow.getAllWindows().length === 0) {
    const mainWindow = createWindow(`http://${webAddr}/?ws=${wsAddr}`, {});
    mainWindow.maximize();
    debug && mainWindow.webContents.openDevTools();
  }
});

app.on('window-all-closed', app.quit);

const ws = new WebSocket(`ws://${wsAddr}/?id=main`);
ws.on('error', e => console.error('main process WebSocket error', e));
ws.on('message', data => {
  let msg = data;
  if (typeof data.toString === 'function') {
    msg = data.toString();
  }
  console.info('main process WebSocket message', msg);
  try {
    msg = JSON.parse(msg);
  } catch (e) {
    console.error('could not parse message', e);
    return;
  }
  try {
    // eslint-disable-next-line no-eval
    eval(msg.data);
  } catch (e) {
    console.error('could not eval javascript', e);
  }
});
ws.on('open', () => console.info('main process WebSocket connected'));
ws.on('close', e => console.info('main process WebSocket closed', e));

app.on('will-quit', () => ws.close(1000, 'app.will-quit'));

app.on('quit', () => ws.send(JSON.stringify({ cmd: 'app.quit' })));

// app.whenReady().then(() => {
//   protocol.registerFileProtocol('file', (request, callback) => {
//     const pathname = decodeURIComponent(request.url.replace('file:///', ''))
//     callback(pathname)
//   })
// })
